<?php

global $cuFormsDisplayed;
$cuFormsDisplayed = array();

class AopsContentUpgrades extends Aops {


	public $pluginRoot = '';
	public $options = array();
	public $cu_id = 0;
	public $jsForHeader = array();        
	public $jsForFooter = array();
	public $htmlForFooter = array();

	public function __construct($pluginRoot) {

		$this->pluginRoot = $pluginRoot;
		$this->options = $this->getOptions();

		/* Register textdomain. */
		add_action('plugins_loaded', array( $this, 'registerTextDomain' ) );

		/* Register style sheet and scripts for the admin area. */
		add_action( 'admin_enqueue_scripts', array( $this, 'registerAdminStylesScripts' ) );

		/* Register style sheet and scripts for the front. */
		add_action( 'wp_enqueue_scripts', array( $this, 'registerFrontStylesScripts' ) );

		/* Register custom post type */
		add_action( 'init', array( $this, 'registerAopsPostType' ) );

		/* Define columns sorting */
		add_filter( 'manage_edit-' . self::$postTypeName . '_sortable_columns', array( $this, 'defineColumnsSorting' ) );

		/* Add custom colums for list of items of our custom post type */
		add_filter('manage_' . self::$postTypeName . '_posts_columns', array( $this, 'registerCustomColumn' ) );

		/* Show custom column for list of items of our custom post type */
		add_action('manage_' . self::$postTypeName . '_posts_custom_column', array( $this, 'showCustomColumn' ), 10, 2);

		/* Process columns sorting */
		add_action( 'pre_get_posts', array( $this, 'processColumnsSorting' ) );

		/* Set up Bulk edit */
		add_action( 'bulk_edit_custom_box', array( $this, 'setBulkEdit'), 10, 2 );

		/* Preload popup on certain pages */
		add_action( 'wp_footer', array($this, 'preloadPopup') );


		/* Place JS needed by plugin */
		add_action( 'wp_head', array($this, 'placeJsInHeader'), 100 );


		/* Place JS & HTML needed for popups */
		add_action( 'wp_footer', array($this, 'placeJsInFooter'), 100 );
		add_action( 'wp_footer', array($this, 'placeHtmlInFooter'), 100 );

		/**** Add shortcodes ****/

		add_shortcode( 'content_upgrade', array( $this, 'processShortcodeCu' ) );
		add_shortcode( 'content_upgrade_link', array( $this, 'processShortcodeCuLink' ) );
		add_shortcode( 'content_upgrade_scripts', array( $this, 'outputCuJavascript' ) );


		/*** AJAX events***/

		add_action( 'wp_ajax_aops_cu_add_entry', array( $this, 'processAjax' ) ); // ajax for logged in users
		add_action( 'wp_ajax_nopriv_aops_cu_add_entry', array( $this, 'processAjax' ) ); // ajax for not logged in users

		/* process Bulk edit request and other AJAX actions  */
		add_action( 'wp_ajax_cu_save_bulk_edit', array( $this, 'processAjax' ) );

		add_action( 'wp_ajax_aops_cu_get_styles', array( $this, 'processAjax' ) );
		add_action( 'wp_ajax_nopriv_aops_cu_get_styles', array( $this, 'processAjax' ) );

	}

	/* Add Options on Activate */
	public static function install() {

		self::installDatabaseTables();

		$optionsVersion = get_option(self::$prefix . 'options_version', "none");
		if ($optionsVersion != self::OPTIONS_VERSION) { // may need to update
			if ($optionsVersion == "none") {
				self::updateOptionsToLatestVersion();
				self::updateCusToLatestVersion();
			}
			else {
				self::updateOptionsToLatestVersion($optionsVersion);
				self::updateCusToLatestVersion($optionsVersion);
			}

		}

		foreach (self::$pluginDefaultOptionValues as $optionId => $defaultValue) {
			// Existing options will not be updated
			// see https://developer.wordpress.org/reference/functions/add_option/
			add_option( self::$prefix . $optionId, $defaultValue);
		}

		self::deleteLog();

	}

	public static function updateOptionsToLatestVersion($currentOptionsVersion = false) {
		if ($currentOptionsVersion) {
			// here will be update process for cases when there are already some options installed
			if (isset(self::$pluginPartialUpdateOptionValues[$currentOptionsVersion])) {
				$newOptions = self::$pluginPartialUpdateOptionValues[$currentOptionsVersion];
				foreach ($newOptions as $optionName => $optionValue) {
					if (is_array($optionValue)) { // array contains instructions what to do if this options already exists
						$currentOptionValue = get_option(self::$prefix . $optionName, "none");

						if ($currentOptionValue == 'none') { // no options currently exists
							update_option(self::$prefix . $optionName, $optionValue[0]); //  $optionValue[0] is default value for fresh install
						}
						else { // option already exists, must decide what to do
							switch  ($optionValue[1]) { //  $optionValue[0] contains instructions what to do if this options already exists
								case 'update_value':
									update_option(self::$prefix . $optionName, $optionValue[2]); //  $optionValue[2] is new value for update
									break;
								case 'keep_current': // do nothing
								default:
									break;
							}
						}
					}
					else { // no instructions what to do if this options already exists , just set a default value
						update_option(self::$prefix . $optionName, $optionValue);
					}

				}
			}
			Aops::log("options were updated from version $currentOptionsVersion to " . self::OPTIONS_VERSION);
		}
		else { // update process for fresh install (or when optios version is unknown)
			foreach (self::$pluginFreshUpdateOptionValues as $optionName => $optionValue) {
				update_option(self::$prefix . $optionName, $optionValue);
			}

			Aops::log("options were updated from fresh state to " . self::OPTIONS_VERSION);
		}
		update_option(self::$prefix . 'options_version', self::OPTIONS_VERSION);
	}

	public static function updateCusToLatestVersion($currentOptionsVersion = false) {
		if ($currentOptionsVersion) {
			// here will be update process for cases when there are already some options installed
		}
		else { // update process for fresh install (or when optios version is unknown)
			global $wpdb;
			$sql = "SELECT ID FROM " . $wpdb->posts . " WHERE post_type = 'aops_content_upgrade'";

			$CUs = $wpdb->get_results($sql);

			foreach ($CUs as $cu) {
				foreach (self::$pluginUpdateMetadataValues as $metaName => $metaValue) {
					update_post_meta($cu->ID, self::$metaPrefix . '_' . $metaName, $metaValue);
				}
			}
		}
	}

	/* Create DB tables for Entries and Logs */
	public static function installDatabaseTables() {
		global $wpdb;
		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

		$charset_collate = $wpdb->get_charset_collate();

		$tableNameEntries = $wpdb->prefix . self::TABLE_ENTRIES;
		$tableNameLogs = $wpdb->prefix . self::TABLE_LOGS;

		$sqlEntries = "CREATE TABLE $tableNameEntries (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			cu_id mediumint(9) NOT NULL,
			name varchar(250) NOT NULL,
			email varchar(250) NOT NULL,
			created_at datetime DEFAULT '0000-00-00 00:00:00' NOT NULL,
			UNIQUE KEY id (id)
		) $charset_collate;";

		dbDelta( $sqlEntries );

		$sqlLogs = "CREATE TABLE $tableNameLogs (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			created_at timestamp NOT NULL,
			data text NOT NULL,
			UNIQUE KEY id (id)
		) $charset_collate;";

		dbDelta( $sqlLogs );

		add_option( self::$prefix . 'db_version', self::$dbVersion );

		Aops::log('Plugin installed');
	}

	public function registerTextDomain() {
		load_plugin_textdomain( 'aops-content-upgrades', false, dirname( plugin_basename( $this->pluginRoot ) ) );
	}

	public function registerAdminStylesScripts($hook) {
		$hooks = array(
			'edit.php',
			'post.php',
			'post-new.php',
			'aops_content_upgrade_page_aops-cu-settings'
		);
		if(!in_array($hook, $hooks)) {
			return;
		}

		wp_enqueue_media();
		wp_enqueue_script( 'jquery' );

		// Tabs
		wp_enqueue_script( 'jquery-easytabs-js', plugins_url( '/js/jquery.easytabs.js' , $this->pluginRoot), array('jquery'), '3.2.0' );

		// Color pickers
		wp_enqueue_script( 'jquery-spectrum-js', plugins_url( '/js/spectrum.js' , $this->pluginRoot), array('jquery'), '1.7.0' );
		wp_enqueue_style( 'spectrum-css', plugins_url( '/css/spectrum.css', $this->pluginRoot ) );

		// Fancybox
		wp_enqueue_script( 'jquery-fancybox2-js', plugins_url( '/js/jquery.fancybox2.js' , $this->pluginRoot), array('jquery'), '2.1.5', true );
		wp_enqueue_style( 'aops-fancybox2', plugins_url( '/css/fancybox2-custom.css', $this->pluginRoot ) );

		//Collapsible
		wp_enqueue_script( 'jquery-collapse-js', plugins_url( '/js/jquery.collapse.min.js' , $this->pluginRoot), array('jquery'), '1.0.0' );

		//Zero Clipboard
		wp_enqueue_script( 'jquery-zeroclipboard-js', plugins_url( '/js/jquery.zclip.min.js' , $this->pluginRoot), array('jquery'), '1.1.1' );

		// Custom styles & scripts for this plugin

		/* disabled on Edit page because AopsContentUpgradeTabs::addTinymceInsertCUPlugin() also adds this script. */
		// if ($hook == 'aops_content_upgrade_page_aops-cu-settings') {
			wp_enqueue_script( 'aops-admin-js', plugins_url( '/js/aops-admin.js' , $this->pluginRoot), array('jquery'), AOPS_CU_VERSION );
		// }

		wp_enqueue_style( 'aops-admin-css', plugins_url( '/css/aops-admin.css', $this->pluginRoot ), array(), AOPS_CU_VERSION );

	}

	public function registerFrontStylesScripts() {

		if ($this->determineWhetherScriptsAreNeeded()) {
			// Register scripts

			wp_enqueue_script( 'jquery' );

			// ----- moved to aops-front.js ---- wp_enqueue_script( 'jquery-fancybox2-js', plugins_url( '/js/jquery.fancybox2.js' , $this->pluginRoot), array('jquery'), '2.1.5', true );
			// ----- moved to aops-front.js ---- wp_enqueue_script( 'jquery-form-validator-js', plugins_url( '/js/jquery.form-validator.min.js' , $this->pluginRoot), array('jquery'), '2.2.201', true );
			// ----- moved to aops-front.js ---- wp_enqueue_script( 'ouibounce-js', plugins_url( '/js/ouibounce.min.js' , $this->pluginRoot), array('jquery'), '0.0.12', true );
			wp_enqueue_script( 'aops-front-js', plugins_url( '/js/aops-front.js' , $this->pluginRoot), array('jquery'), AOPS_CU_VERSION, true );

			$cuVars = array(
				'ajaxurl'													=> admin_url('admin-ajax.php'),
				'disableClickEventListener'				=> $this->options['disable_event_listener'],
				'html5ValidationEnabled'					=> $this->options['enable_html5_validation'],
				'defaultJQueryVersionEnabled'			=> $this->options['enable_default_jquery_version']
			);

			wp_localize_script( 'aops-front-js', 'cu_vars', $cuVars ); // set CU variables in JS file

			wp_localize_script( 'aops-front-js', 'ajax_object', array(
				'ajaxurl' => admin_url( 'admin-ajax.php' ),
			));

			// Register styles
			if (!$this->options['css_disabled']) {
				wp_enqueue_style( 'aops-front', plugins_url( '/css/aops-front.css', $this->pluginRoot, array(), AOPS_CU_VERSION ) );
				wp_enqueue_style( 'aops-fancybox2', plugins_url( '/css/fancybox2-custom.css', $this->pluginRoot ), array(), AOPS_CU_VERSION );

				// Custom CSS generated by User settings
				$customCSS = $this->generateCustomCSS();
				wp_add_inline_style( 'aops-front', $customCSS);

				// Custom CSS entered by user
				if (trim($this->options['css_custom_content'])) {
					$userCSS = $this->options['css_custom_content'];
					wp_add_inline_style( 'aops-front', $userCSS);
				}
			}
		}
	}

	public function determineWhetherScriptsAreNeeded() {

		$result = false;

		// We can reliably determine necessity of CU JS scripts
		// only on the single pages or posts.
		// Because any other page (e.g. main page or list of category posts)
		// may include any number of posts in any combinations
		// and there are no easy way to check each and every one of those posts
		// before outputting page content.

		// So we determine which page we are on (single post, main page, category page, archives)
		// and if it's not single and plugin settings are enabling this page, then just add scripts anyway.
		if (is_single() || is_page()) {
			global $post;
			$cuIds = array();

			if (is_object($post)) {
				$template = get_post_meta( $post->ID, '_wp_page_template', true );
				$templateMatch = false;

				if ($template) { // this is a page
					if (isset($this->options['load_cu_on_a_template'])) {
						if ($this->options['load_cu_on_a_template'] == $template) {
							$templateMatch = true;
						}
					}
				}
				else { // this is a post, don't add CU scripts until determined that there are any CUs inserted
					$templateMatch = false;
				}

				$cuIds = $this->findPopupsToPreload($post->ID); // try to find CUs that need to be preloaded for this post or page.
				$preloadFound = ((is_array($cuIds)) && (count($cuIds) > 0 ));

				$shortcodeFound = $this->checkContentForShortcode($post); // try to find CU shortcode in this post or page.
			}

			if ($shortcodeFound || $preloadFound || $templateMatch) {
				$result = true;
			}
		}

		if ( ! $result ) {
			if (is_front_page() || is_home()) {
				$result = isset($this->options['load_cu_main_page']) ?
					(bool) $this->options['load_cu_main_page'] : false;
			}
			elseif (is_category()) {
				$result = isset($this->options['load_cu_category_page']) ?
					(bool) $this->options['load_cu_category_page'] : false;
			}
			elseif (is_date()) {
				$result = isset($this->options['load_cu_date_page']) ?
					(bool) $this->options['load_cu_date_page'] : false;
			}
		}

		return $result;
	}

	public function checkContentForShortcode($post) {
		// Check for [content_upgrade] shortcode

		$regexp = '/\[content_upgrade(.*)\[\/content_upgrade\]/i';
		$post_id = $post->ID;

		if ($post->post_type == 'aops_landing_page') {
			$lpPrefix = 'aops_lp_meta_';

			$masthead			= get_post_meta($post_id, $lpPrefix . 'masthead_text', true);
			$actionTextTop		= get_post_meta($post_id, $lpPrefix . 'action_text_top', true);
			$bodyContent		= get_post_meta($post_id, $lpPrefix . 'body_content', true);
			$actionTextBottom	= get_post_meta($post_id, $lpPrefix . 'action_text_bottom', true);
			$footerContent		= get_post_meta($post_id, $lpPrefix . 'footer_text', true);

			$resultMasthead		= preg_match($regexp, $masthead, $matches);
			$resultActionTop	= preg_match($regexp, $actionTextTop, $matches);
			$resultBody			= preg_match($regexp, $bodyContent, $matches);
			$resultActionBottom	= preg_match($regexp, $actionTextBottom, $matches);
			$resultFooter		= preg_match($regexp, $footerContent, $matches);

			$result = $resultMasthead || $resultActionTop || $resultBody || $resultActionBottom || $resultFooter;

		}
		else {
			$resultFieldContent = false;
			$resultPostContent = preg_match($regexp, $post->post_content, $matches);

			if ( function_exists( 'get_fields' ) ) { // check for ACF installed
				if ( $this->options['check_custom_fields_for_cus'] ) {
					$resultFieldContent = $this->checkCustomFieldsForCUs( $post_id );
				}
			}

			$result = $resultPostContent || $resultFieldContent;
		}
/*
		self::log(
			array(
				'function' => 'checkContentForShortcode',
				'id' => $post->ID,
				'type' => $post->post_type,
				'result' => '[' . $result . ']'
			)
		);
*/
		return (bool)$result;
	}

	public function checkCustomFieldsForCUs( $post_id ) {
		$cuFound = false;

		$foundFields = get_fields( $post_id );

		if ( is_array( $foundFields ) ) {
			foreach ( $foundFields as $fieldContent ) {
				if ( is_array( $fieldContent ) && isset( $fieldContent[0] ) ) { // this is a repeater field, or "flexible content" field
					foreach ( $fieldContent[0] as $fieldsSubContent) {
						$cuFound = $this->checkContentForCU( $fieldsSubContent );
						if ( $cuFound ) break;
					}
					if ( $cuFound ) break;
	 			}
				else {
					$cuFound = $this->checkContentForCU( $fieldContent );
					if ( $cuFound ) break;
				}
			}
		}

		return $cuFound;
	}

	public function checkContentForCU( $fieldContent ) {

		$cuFound = false;

		if ( is_string( $fieldContent ) ) {
			$regexp = '/\[content_upgrade(.*)\[\/content_upgrade\]/i';
			$cuFoundShortcode = preg_match($regexp, $fieldContent, $matches);
			$cuFoundOutput = ! ( strpos( $fieldContent, 'class="cu-button' ) === false );
			$cuFound = $cuFoundShortcode || $cuFoundOutput;
		}
		return $cuFound;
	}

	public function registerAopsPostType() {
		$labels = array(
			'name' 				=> _x( 'Content Upgrades', 'post type general name' ),
			'singular_name'		=> _x( 'Content Upgrade', 'post type singular name' ),
			'add_new' 			=> __( 'Add New Content Upgrade', 'aops' ),
			'add_new_item' 		=> __( 'Content Upgrade Item', 'aops' ),
			'edit_item' 		=> __( 'Edit Content Upgrade', 'aops' ),
			'new_item' 			=> __( 'New Content Upgrade', 'aops' ),
			'view_item' 		=> __( 'View Content Upgrade', 'aops' ),
			'search_items' 		=> __( 'Search Content Upgrade', 'aops' ),
			'not_found' 		=> __( 'No Content Upgrade Found', 'aops' ),
			'not_found_in_trash'=> __( 'No Content Upgrade Found In Trash', 'aops' ),
			'parent_item_colon' => __( 'Parent Content Upgrade', 'aops' ),
			'menu_name'			=> __( 'Content Upgrades', 'aops' )
		);

		$taxonomies = array();
		$supports = array( 'title' );

		$post_type_args = array(
			'labels' 			=> $labels,
			'singular_label' 	=> __( 'Content Upgrade', 'aops' ),
			'public' 			=> false,
			'show_ui' 			=> true,
			'publicly_queryable'=> true,
			'query_var'			=> true,
			'capability_type' 	=> 'post',
			'has_archive' 		=> false,
			'hierarchical' 		=> false,
			'supports' 			=> $supports,
			'menu_position' 	=> 20,
			'taxonomies'		=> $taxonomies,
			'menu_icon'			=> 'dashicons-email-alt'
		);

		register_post_type( self::$postTypeName, $post_type_args );
	}


	// used as fake callback for add_settings_field() in $this->registerSettings()
	// callback is not used because fields are rendered by $this->displaySettingsPage()
	public function dummy() {

	}

	/**
	 * Defines columns sorting
	 * @param array $columns
	 * @return array
	 */
	public function defineColumnsSorting( $columns ) {
		$columns['aops_id'] = 'id';
		return $columns;
	}

	/**
	 * Processes sorting of custom columns
	 * @param WP_Query $query
	 */
	public function processColumnsSorting( $query ) {
		if( ! is_admin() )
			return;

		$orderby = $query->get( 'orderby');
		switch ($orderby) {
			case 'aops_id':
				$query->set('orderby','ID');
				break;
		}
	}


	// add custom column to the Cu posts list
	function registerCustomColumn($defaults) {
		$defaults['aops_id']			= __('ID', 'aops');
		$defaults['cu_entries_number']	= __('Entries', 'aops');
		$defaults['cu_email_status']	= __('Emails', 'aops');
		return $defaults;
	}

	// show custom column in the CU posts list
	function showCustomColumn($column_name, $post_ID) {
		switch ( $column_name ) {
			case 'aops_id' :
				echo $post_ID;
				break;
			case 'cu_entries_number':
				$entriesNumber = AopsEntryManager::getCuEntriesNumber($post_ID);
				if ($entriesNumber) {
					echo $entriesNumber;
				}
				else {
					echo '0';
				}
				break;
			case 'cu_email_status':
				echo $this->getCuEmailStatus($post_ID) ? 'On' : 'Off';
				break;
		}
	}


	public function getCuEmailStatus($cuPostId) {
		$emailStatus = get_post_meta( $cuPostId, self::$metaPrefix . '_user_email_enabled', true );
		return $emailStatus;
	}


	public function setBulkEdit( $column_name, $post_type ) {
		if ($post_type == self::$postTypeName) {
			switch ($column_name) {
				case 'cu_email_status':
					$out = '<fieldset class="inline-edit-col-right">';
					foreach (self::$bulkEditCheckboxes as $checkboxName => $field) {
						$out .= <<<EOT
						<div class="inline-edit-group cu-bulk-edit-group">
							<label>
								<span class="title cu-bulk-edit-label">{$field['title']}</span>
								<select name="{$checkboxName}" class="{$checkboxName}">
									<option value="-1">— No Change —</option>
									<option value="0">On</option>
									<option value="1">Off</option>
								</select>
							</label>
						</div>
EOT;
					}
					$out .= '</fieldset>';
					echo $out;
				break;
			}
		}
	}

	/**
	 * Assebmles popup content for content_upgrade shortcode
	 * @param integer $cuPostId ID of 'aops_content_upgrade' post
	 * @return string
	 */
	public function prepareContentPopup($cuPostId) {
		$meta = $this->getMeta($cuPostId);
		$opts = $this->options;

		if (!$meta['button_text']) {
			$meta['button_text'] = $opts['button_default_text'];
		}

		$out = '<div class="cu-popup-container" id="cu_' . $cuPostId . '" style="display:none;">';
		$out .= '<div class="cu-form-container">';

		if ($meta['headline']) {
			$out .= '<h2 class="cu-popup-headline" >'. $meta['headline'] . '</h2>';
		}

		if ($meta['description']) {
			$out .= '<div class="cu-popup-description">' . nl2br($meta['description']) . '</div>';
		}

		if ($opts['enable_html5_validation']) {
			$email_field = "<div class=\"cu-popup-field-wrap\"><input type=\"email\" name=\"cu_popup_email\" data-validation=\"email\" data-validation-error-msg=\"Please enter a valid email address\" placeholder=\"Email\" value=\"\" required /></div>";
		}
		else {
			$email_field = "<div class=\"cu-popup-field-wrap\"><input name=\"cu_popup_email\" data-validation=\"email\" data-validation-error-msg=\"Please enter a valid email address\" placeholder=\"Email\" value=\"\"/></div>";
		}

		if ($opts['consent_checkbox_enabled']) {
			$consentLabel = $opts['consent_checkbox_label'] ? $opts['consent_checkbox_label'] : 'I consent to my contact information being collected.';
			$consent_checkbox = "<div class=\"cu-consent-chkbox\"><input type=\"checkbox\" id=\"cu-consent-$cuPostId\" name=\"cu_consent\" value=\"1\"><label for=\"cu-consent-$cuPostId\" class=\"cu-consent-label\">{$consentLabel}</label></div>";
		}
		else {
			$consent_checkbox = '';
		}

		$out .= "<form class=\"cu-popup-form\" id=\"cu-popup-form-".$cuPostId."\" method=\"POST\" name=\"cu_popup_form\" action=\"/wp-content/plugins/content-upgrades/fallback.php\" >"
			. wp_nonce_field( 'cu-popup-form', 'cu-popup-form-nonce', false )
			. "<input type=\"hidden\" name=\"cu_popup_id\" value=\"{$cuPostId}\"/>"
			. "<div class=\"cu-popup-field-wrap\"><input name=\"cu_popup_firstname\" placeholder=\"First Name\" value=\"\"/></div>"
			. $email_field
			. $consent_checkbox
			. "<button class=\"cu-popup-button\">{$meta['button_text']}<span class=\"cu-popup-loading\"></span></button>"
			. "<div class=\"cu-popup-note\">No spam. Unsubscribe anytime.</div>"
		. "</form>";

		$out .= '</div></div>';
		$out .= '<div id="flash-message" style="z-index:10000; display:none;"></div>';

		return $out;
	}


	/**
	 * Composes JS code needed for Drip integration and Facebook conversion event
	 * @param type $cuPostId
	 */
	public function prepareContentJS($cuPostId) {
		$meta = $this->getMeta($cuPostId);
		$options = $this->getOptions();

		$cuTitle = get_the_title($cuPostId);
		$cuRawTitle = get_post_field( 'post_title', $cuPostId, 'raw' );

		/*** (1) ***
		 *	If any tags (comma separated) are entered in "CU Settings > Integrations > Drip > Default Tags",
		 *	these should always be assigned to the subscriber in Drip,
		 *	in addition to the CU-specific tag(s) that are defined when editing an individual CU.
		 */

		if ($options['drip_default_tags']) {
			$defaultTags = explode(',', $options['drip_default_tags']);
		}
		else {
			$defaultTags = array();
		}

		if (trim($meta['drip_tag']) == '') {
			// set empty array of CU tags since "Drip Tags" field in CU setting is empty
			$cuTags	= array();
		}
		else {
			// "Drip Tags" field in CU setting is not empty, use it to set default array of tags
			$cuTags = explode(',', trim($meta['drip_tag']));
		}

		// Create auto tag. Later will determine whether it should be applied
		$cuAutoTag = '[Content upgrade] ' . htmlspecialchars($cuTitle);

		// If the checkbox for automatic tagging is checked,
		// then it should automatically apply a tag, "[Content Upgrade] CU Name"

		if ($meta['drip_tag_enabled']) {
			$cuTags[] = $cuAutoTag;
		}

		// see comment at (1)
		$finalCuTags = array_merge($defaultTags, $cuTags);

		if (count($finalCuTags)) {

			// turn PHP tags array into JSON tags array

			$json_drip_tags = "[";
			$sep = '';
			foreach ($finalCuTags as $drip_tag) {
				if ($drip_tag != "") {
					$json_drip_tags .= $sep . '"' . htmlspecialchars(trim($drip_tag)) . '"';
					$sep = ",";
				}
			}
			$json_drip_tags .= ']';
		}
		else {
			$json_drip_tags = '"0"'; // Special case later checked in aops-front.js
		}

		if ($options['drip_global_event']) {
			$globalEvent = trim($options['drip_global_event']);
		}
		else {
			$globalEvent = 0;
		}

		$drip_custom_event_name = json_encode($meta['drip_custom_event_name']);

		$global_facebook_conversion_event_enabled = ($options['facebook_conversion_event_enabled']) ? 1 : 0;
		$global_facebook_conversion_event_name = json_encode($options['facebook_conversion_event_name']);

		$custom_facebook_conversion_event_enabled = ($meta['facebook_conversion_event_enabled']) ? 1 : 0;
		$custom_facebook_conversion_event_name = json_encode(str_replace('{content_upgrade_title}', $cuRawTitle, $meta['facebook_conversion_event_name']));

		$google_analytics_submit_event_enabled = ($meta['google_analytics_submit_event_enabled']) ? 1: 0;

		$global_conversion_event_enabled = ($options['global_conversion_event_enabled']) ? 1: 0;
		$global_conversion_event_code = json_encode($options['global_conversion_event_code']);

		$custom_conversion_event_enabled = ($meta['custom_conversion_event_enabled']) ? 1: 0;
		$custom_conversion_event_code = json_encode($meta['custom_conversion_event_code']);

		$redirect_after_formfill_enabled = ($meta['redirect_after_formfill_enabled']) ? 1 : 0;
		$redirect_after_formfill_url = json_encode($meta['redirect_after_formfill_url']);

		// Handle download popup text
		$download_text = json_encode($this->prepareDownloadText($meta, $options, $cuRawTitle));

		$download_enabled	= $meta['download_enabled']		? 1 : 0;

		if ($this->recentCuSubmissionByThisUser($cuPostId) || $this->thisIsAnArchivePage()) {
			$exit_popup_enabled	=  0;
		}
		else {
			$exit_popup_enabled	= $meta['exit_popup_enabled']	? 1 : 0;
		}

		$out = <<<EOT
			<script type="text/javascript">
				if (typeof(content_upgrades_opts) === "undefined") {
					var content_upgrades_opts = [];
				}

				content_upgrades_opts[{$cuPostId}] = {
					"global_event_name": "{$globalEvent}",
					"custom_event_enabled": "{$meta['drip_custom_event_enabled']}",
					"custom_event_name": {$drip_custom_event_name},
					"custom_name_field": "{$meta['drip_first_name_field_id']}",
					"drip_tag": {$json_drip_tags},

					"global_facebook_conversion_event_enabled": {$global_facebook_conversion_event_enabled},
					"global_facebook_conversion_event_name": {$global_facebook_conversion_event_name},
					"custom_facebook_conversion_event_enabled": {$custom_facebook_conversion_event_enabled},
					"custom_facebook_conversion_event_name": {$custom_facebook_conversion_event_name},

					"google_analytics_submit_event_enabled": {$google_analytics_submit_event_enabled},

					"global_conversion_event_enabled": {$global_conversion_event_enabled},
					"global_conversion_event_code": {$global_conversion_event_code},
					"custom_conversion_event_enabled": {$custom_conversion_event_enabled},
					"custom_conversion_event_code": {$custom_conversion_event_code},

					"cu_title": "{$cuAutoTag}",

					"download_popup_enabled": {$download_enabled},
					"download_popup_text": {$download_text},

					"redirect_after_formfill_enabled" : {$redirect_after_formfill_enabled},
					"redirect_after_formfill_url" : {$redirect_after_formfill_url},

					"exit_popup_enabled": {$exit_popup_enabled}
				};
			</script>
EOT;
		return $out;

	}

	public function thisIsAnArchivePage() {
		if (is_front_page() || is_home() || is_category() || is_date() ) {
			return true;
		}
		return false;
	}

	public function recentCuSubmissionByThisUser($cuID) {
		$cookieWasSubmitted = AopsCookieManager::getCuSubmissionCookie($cuID);
		if ($cookieWasSubmitted) {
			$result = (time() - $cookieWasSubmitted) < DAY_IN_SECONDS ? true : false;
		}
		else {
			$result = false;
		}
		return $result;
	}


	public function prepareDownloadText($meta, $options, $contentUpgradeTitle) {
		$downloadText = $meta['download_text'] ? $meta['download_text'] : $options['download_text'];
		$text = wpautop($downloadText);

		$replacements = array(
			array(
				'search'	=> '{bonus_content_title_link}',
				'replace'	=> $meta['bonus_content_path']?
					'<a class="cu-popup-button download-link" href="{bonus_content_url}" target="_blank">{content_upgrade_title}</a>': "[No link provided]"
			),
			array(
				'search'	=> '{content_upgrade_title}',
				'replace'	=> $contentUpgradeTitle
			),
			array(
				'search'	=> '{bonus_content_url}',
				'replace'	=> $meta['bonus_content_path']
			)
		);

		foreach ($replacements as $replacement) {
			$text = str_replace($replacement['search'], $replacement['replace'], $text);
		}

		if ($meta['share_buttons_enabled']) {
			$text .= $this->displayShareButtons($meta['share_buttons_header_text'], $options);
		}
		return $text;

	}

	public function displayShareButtons($headerText, $options) {
		if (!$headerText) {
			$headerText = $options['share_buttons_header_text'];
		}

		$twitterButton = '<a href="https://twitter.com/share" class="twitter-share-button" data-show-count="false">Tweet</a>';
		$twitterScript = '<script async src="//platform.twitter.com/widgets.js" charset="utf-8"></script>';

		// Note: js will process {{url}} and {{encoded_url}} here and replace them with the current URL
		$facebookButton = '<div class="fb-share-button" data-href="{{url}}" data-layout="button" data-mobile-iframe="true"><a class="fb-xfbml-parse-ignore" target="_blank" href="https://www.facebook.com/sharer/sharer.php?={{encoded_url}}&amp;src=sdkpreparse">Share</a></div>';
		$facebookScript = <<<EOT
<div id="fb-root"></div>
<script>(function(d, s, id) {
  var js, fjs = d.getElementsByTagName(s)[0];
  if (d.getElementById(id)) return;
  js = d.createElement(s); js.id = id;
  js.src = "//connect.facebook.net/en_US/sdk.js#xfbml=1&version=v2.6";
  fjs.parentNode.insertBefore(js, fjs);
}(document, 'script', 'facebook-jssdk'));</script>
EOT;
		$linkedinButton = '<script src="//platform.linkedin.com/in.js" type="text/javascript"> lang: en_US</script><script type="IN/Share"></script>';

		$text = <<<EOT
			<div class="sharing-buttons">
				<p class="sharing-label">{$headerText}</p>

				<div>
					{$twitterButton}
					{$facebookButton}
					{$linkedinButton}
				</div>

				{$twitterScript}
				{$facebookScript}
			</div>
EOT;
		return $text;
	}

	/**
	 * Adds JS
	 * @param string $js
	 */
	public function addJsInHeader($js) {
		$this->jsForHeader[] = $js;
	}

	/**
	 * Adds JS
	 * used by preloadPopup() and processShortcodeCu()
	 * see placeJsInFooter()
	 * @param string $js
	 */
	public function addJsInFooter($js) {
		$this->jsForFooter[] = $js;
	}

	/**
	 * Adds HTML
	 * used by processShortcodeCu()
	 * see placeHtmlInFooter()
	 * @param type $html
	 */
	public function addHtmlInFooter($html) {
		$this->htmlForFooter[] = $html;
	}

	/**
	 * Called by "wp_head" action
	 */
	public function placeJsInHeader() {
		foreach ($this->jsForHeader as $js) {
			echo $js;
		}
	}

	/**
	 * Called by "wp_footer" action
	 */
	public function placeJsInFooter() {
		foreach ($this->jsForFooter as $js) {
			echo $js;
		}
	}

	/**
	 * Called by "wp_footer" action
	 */
	public function placeHtmlInFooter() {
		foreach ($this->htmlForFooter as $html) {
			echo $html;
		}
	}

	/**
	 * Preloads popup(s) if needed by some page
	 * @global type $post
	 */
	public function preloadPopup()
	{
		global $post;
		if ( is_object( $post ) ) {
			if ( !is_admin() ) { // TODO: why there is a check for admin? maybe remove
				$cuIds = $this->findPopupsToPreload($post->ID);
				if (is_array($cuIds) && (count($cuIds) > 0)) {
					foreach ($cuIds as $cuPostId) {
						echo $this->prepareContentPopup($cuPostId);
						$js = $this->prepareContentJS($cuPostId);
						$this->addJsInFooter($js);
					}
				}
			}
		}
	}

	/**
	 * Determines which content upgrades need a popup for a given page
	 * @param integer $pageId
	 * @return array
	 */
	public function findPopupsToPreload($pageId) {

		if (!$pageId) return false;

		$cuPostIds = array();

		$args = array(
			'post_type' => self::$postTypeName,
			'meta_query' => array(
				array(
					'key'     => self::$metaPrefix . '_' . 'preload_page_ids',
					'value'   => $pageId,
					'compare' => 'LIKE',
				)
			)
		);

		$query = new WP_Query( $args );

		if ( $query->have_posts() ) {
			while ( $query->have_posts() ) {
				$query->the_post();
				$cuPostId = get_the_ID();
				$meta = get_post_meta( $cuPostId, self::$metaPrefix . '_' . 'preload_page_ids', true );
				$preloadPageIds = explode(',', $meta);
				foreach ($preloadPageIds as $preloadPageId) {
					if (intval(trim($preloadPageId)) == $pageId) {
						$cuPostIds[] = $cuPostId;
					}
				}
			}
		}
		wp_reset_postdata();
		return $cuPostIds;
	}

	/**
	 * Shortcode for text block and button
	 */
	function processShortcodeCu( $atts, $content = null ) {
		global $cuFormsDisplayed;
		global $post;
		extract( shortcode_atts( array(
			'id' => -1,
			'cu_id' => -1,
			'preload' => -1
		), $atts ) );

		if ($id > 0) {
			$cId = $id;
		}
		elseif ($cu_id > 0) {
			$cId = $cu_id;
		}
		else {
			$cId = 0;
		}


		if ($cId) {
			$args = array(
				'post_type'	=> self::$postTypeName,
				'p'			=> $cId,
				'status'	=> array( 'publish' )
			);

			$cu_posts = get_posts( $args );

			if ( count( $cu_posts ) ) {

				$cu_post = array_pop( $cu_posts );
				$out = '';

				$cu_post_id = $cu_post->ID;

				$out .= $this->displayContentAndButton( $cu_post_id, $content );

				$cuIds = $this->findPopupsToPreload( $post->ID );
				$preloadFound = is_array($cuIds) && (count($cuIds) > 0 );
				$currentCUfoundInPreload = $preloadFound ? in_array($cu_post_id, $cuIds) : false;


				if ( $currentCUfoundInPreload ) {
					// do not output content popup HTML or JS as is will be added by preload`
				}
				else if ( $preload != -1 ) {
					$popupHtml = $this->prepareContentPopup( $cu_post_id );
					$contentJs = $this->prepareContentJS( $cu_post_id );

					echo $contentJs;
					echo $popupHtml;
				}
				else if ( ! in_array( $cu_post_id, $cuFormsDisplayed ) ) {
					$popupHtml = $this->prepareContentPopup( $cu_post_id );
					$contentJs = $this->prepareContentJS( $cu_post_id );
					$this->addJsInFooter( $contentJs );
					$this->addHtmlInFooter( $popupHtml );

					$cuFormsDisplayed[] = $cu_post_id;
				}
			}
			else {
				$out = "[ Content upgrade with ID = $cId not found ]";
			}
		}
		else {
			$out = "[ Content upgrade ID not specified ]";
		}
		return $out;
	}


	/**
	 * Displays shortcode contents and button
	 * It is called by $this->processShortcodeCu()
	 *
	 * This function checks if content contains [content_upgrade_button] shortcode
	 * The button should not display in the call-to-action by default.
	 * Only display the button IF the [content_upgrade_button]...text...[/content_upgrade_button] is present.
	 *
	 * [content_upgrade_button] is not a registered wordpress shortcode and cannot be used with do_shortcode()
	 */
	function displayContentAndButton($postID, $content) {

		// Make any link (A tag) inside of the [content_upgrade] ... [/content_upgrade]
		// launch the popup for this content upgrade.
		$content = $this->processContentLinks($postID, $content);

		// Check if [content_upgrade] shortcode text contains button shortcode
		if (preg_match('/\[content_upgrade_button\](.*)\[\/content_upgrade_button\]/i', $content, $matches)) {

			// Ensure that the button text
			// specified inside of the [content_upgrade_button]...text...[/content_upgrade_button]
			// is the text that displays on the button.
			$buttonText = $matches[1];

			// Remove button shortcode from the [content_upgrade] text
			$content = str_replace($matches[0], '', $content);

			$html = "<div class=\"cu-wrapper\">"
					. "<div class=\"cu-content\">{$content}</div>"
					. "<div class=\"cu-button et_smooth_scroll_disabled\">"
						. "<a class=\"cu-popup button button-primary button-large\" href=\"#cu_{$postID}\">{$buttonText}</a>"
					. "</div>"
				. "</div>";
		}
		else {
			$html  = "<div class=\"cu-wrapper\">"
						. "<div class=\"cu-content-wide\">{$content}</div>"
					. "</div>";
		}

		return $html;
	}

	/**
	 * Processing function for content of [content_upgrade_button]
	 * It is called by $this->processShortcodeCuButton()
	 *
	 * Makes any link (A tag) inside of the [content_upgrade] ... [/content_upgrade]
	 * to launch the popup for this content upgrade.
	 *
	 */
	function processContentLinks($postID, $content) {

		$popupUrl = '#cu_' . $postID;
		$popupClass = 'cu-popup';

		// 1. Replace HREF in all links

		$pattern = "/(?<=href=(\"|'))[^\"']+(?=(\"|'))/";
		$content = preg_replace($pattern, $popupUrl, $content);

		// 2. Add CSS class for all links

		$offsetStart = 0;
		$i = 0;
		$newContent = $content;

		while (($offsetStart !== false) && (++$i < 20)) {
			$offsetStart = strpos($content, '<a', $offsetStart + 2);
			$offsetEnd = strpos($content, '/a>', $offsetStart + 2);


			$link = substr($content, $offsetStart, $offsetEnd - $offsetStart  + 3);

			if (strpos($link, "href=") === false) {
				$oldLink = str_replace("<a", "<a href='" . $popupUrl . "'", $link);
			}
			else {
				$oldLink = $link;
			}

			$classAdded = false;
			if (strpos($link, "class='") !== false) {
				$newLink = str_replace("class='", "class='" . $popupClass . ' ', $oldLink);
				$newContent = str_replace($link, $newLink, $newContent);
				$classAdded = true;
			}
			elseif (strpos($link, "class=\"") !== false) {
				$newLink = str_replace("class=\"", "class=\"" . $popupClass . ' ', $oldLink);
				$newContent = str_replace($link, $newLink, $newContent);
				$classAdded = true;
			}

			if (!$classAdded) {
				$newLink = str_replace("href=", "class='" . $popupClass . "' href=", $oldLink);
				$newContent = str_replace($link, $newLink, $newContent);
			}

		}

		return $newContent;
	}


	/**
	 * Processing function for [content_upgrade_link] shortcode
	 * called by do_shortcode()
	 */
	public function processShortcodeCuLink( $atts, $content = null ) {
		global $cuFormsDisplayed;
		extract( shortcode_atts( array(
			'id' => -1,
			'popup' => false
		), $atts ) );

		$args = array(
			'post_type'	=> self::$postTypeName,
			'p'			=> $id,
			'status'	=> array( 'publish' )
		);

		$query = new get_posts( $args );
		if ( count($posts) ) {

			$post = array_pop($posts);
			$out = '';
			$post_id = $post->id;

			$content .= '<a class="cu-popup" href="#cu_' . $post_id .'">' .$content .'</a>';

			if ($popup) { // $popup is defined in shortcode atts
				if (!in_array($post_id, $cuFormsDisplayed)) { // to avoid duplication
					$popupHtml = $this->prepareContentPopup($post_id);
					$contentJs = $this->prepareContentJS($post_id);

					$this->addJsInFooter($contentJs);
					$this->addHtmlInFooter($popupHtml);

					$cuFormsDisplayed[] = $post_id;
				}
			}
		}
		else {
			$content = "[ Content upgrade with ID = $id not found ]";
		}

		return $content;
	}


	/**
	 * Processing function for [content_upgrade_scripts] shortcode
	 * called by do_shortcode()
	 */
	public function outputCuJavascript( $atts, $content = null ) {
		$filename = dirname( __FILE__ ) . DIRECTORY_SEPARATOR . '..' . DIRECTORY_SEPARATOR . 'js' . DIRECTORY_SEPARATOR . 'aops-front.js';

		// need to manually insert "cu_vars" and "ajaxurl" in case if original localized aops-front didn't load

		$localizedVars = "var cu_vars = " . wp_json_encode( array(
			'ajaxurl'					=> admin_url('admin-ajax.php'),
			'disableClickEventListener'	=> $this->options['disable_event_listener'],
			'html5ValidationEnabled' => $this->options['enable_html5_validation']
		) ) . ';';

		$localizedVars .= "var ajax_object = " . wp_json_encode( array(
			'ajaxurl' => admin_url( 'admin-ajax.php' ),
		) ) . ';';

		$fileContents = file_get_contents( $filename );

		return '<script type="text/javascript">' . $localizedVars . ' ' .  $fileContents . '</script>';
	}

	public function processAjax($return_result = false) {
		
		$input = $_POST;
		$result = '';
		$errorUnknownAction = '{"result":"fail", "reason":"unknown action"}';
		$errorInsufficuentData = '{"result":"fail", "reason":"insufficient data"}';

//		Aops::log('processAjax');
//		Aops::log($input);

		if (isset($input['action'])) { // handle AJAX data request
			if (isset($input['data'])) {
				$data = $input['data'];
				$cuId = isset($data['cu_id']) ? $data['cu_id'] : false;
				$cuIds = isset($data['cu_ids']) ? $data['cu_ids'] : false;

				switch ($input['action']) {
					case 'aops_cu_get_info':
						$cuId = isset($data['cu_id']) ? $data['cu_id'] : false;
						if ($cuId) {
							$cuInfo = $this->prepareContentJS($cuId);
							$result = '{"result":"ok", "data":"' . $cuInfo . '"}';
						}
						else {
							$result = $errorInsufficuentData;
						}
						break;
					case 'aops_cu_get_styles':
						$cuStyles = $this->generateCustomCSS();
						// Custom CSS entered by user

						if (trim($this->options['css_custom_content'])) {
							$userCSS = $this->options['css_custom_content'];
							$cuStyles .= $userCSS;
						}
						$result = '{"result":"ok", "data":"' . $cuStyles . '"}';
						break;
					case 'aops_cu_add_entry':
						// handle form submission
						check_ajax_referer( 'cu-popup-form', 'security' );
						$result = $this->processPostData($data);
						break;
					case 'cu_save_bulk_edit':
						check_ajax_referer( 'bulk-posts', 'security' );
						if ( current_user_can( 'edit_posts' ) ) {
							$result = $this->processBulkEdit($data);
						}
						break;
					default:
						$result = $errorUnknownAction;
				}
			}
			else {
				$result = $errorInsufficuentData;
			}
		}

		if ($return_result) {
			return $result;
		}
		else {
			echo $result;
		}
		die();
	}

	function processPostData($input) {
		if ( isset($input['cu_id']) && isset($input['firstname']) && isset($input['email'])) {
			$cuID = intval($input['cu_id']);
			$firstname = sanitize_text_field($input['firstname']);
			$email = sanitize_email(trim($input['email']));
			$date = date('Y-m-d H:i:s',time());

			$entry = new AopsEntry($cuID, $firstname, $email, $date);
			$entryManager = new AopsEntryManager();
			if ($entryManager->checkForDuplicates($entry)) {
				$entryManager->save($entry);

				$emailManager = new AopsEmailManager($cuID);
				$emailManager->sendEmails($entry);
				self::log("AopsMailchimpIntegration::processSubmission($cuID, $email, $firstname); ");
				AopsMailchimpIntegration::processSubmission($cuID, $email, $firstname);
				AopsConvertkitIntegration::processSubmission($cuID, $email, $firstname);
				AopsZapierIntegration::processSubmission($cuID, $email, $firstname);
				AopsCookieManager::setCuSubmissionCookie($cuID);
			}
			else {
				self::log("AopsEmailManager found a duplicate ($cuID, $email, $firstname); ");
			}

			return 'ok';
		}
		else {
			return  'fail';
		}
	}

	public function processBulkEdit($input) {
		if ( isset($input['cu_ids'])) {
			foreach (self::$bulkEditCheckboxes as $fieldName => $field) {
				if (isset($input[$fieldName]) && $input[$fieldName] != '-1') {
					$newStatus = $input[$fieldName] ? 0 : 1;

					foreach ($input['cu_ids'] as $cuId) {
						update_post_meta($cuId, self::$metaPrefix . '_' . $field['name'] , $newStatus);
					}
				}
			}
			return 'ok';
		}
		else {
			return 'fail';
		}
	}

	// Used to process custom CSS options entered by used in the Settings page
	public function generateCustomCSS() {
		$opts = $this->options;

		//---------------- Start Box styles
		$css = ".cu-wrapper { ";

				// border style
				$css .= "border: "
					. $opts['box_border_width'] . " "
					. $opts['box_border_style'] . " "
					. $opts['box_border_color'] . "; ";

				//  colors
				$css .= "color: " . $opts['box_text_color'] . "; "
					. "background-color: " . $opts['box_bg_color'] . "; ";

		// end Box styles
		$css .= '}';

		//---------------- Start Button styles
		$css .= ".cu-button a.button { ";

				// Button colors
				$css .= "color: " . $opts['btn_text_color'] . "; "
					. "background-color: " . $opts['btn_bg_color'] . "; ";

		// end Button styles
		$css .= '}';

		//---------------- Start Hover Button styles
		$css .= ".cu-wrapper a.button:hover,
				.cu-wrapper a.button:focus,
				.cu-wrapper a.button:active { ";

				// colors
				$css .= "color: " . $opts['btn_hover_text_color'] . "; "
					. "background-color: " . $opts['btn_hover_bg_color'] . "; ";

		// end Hover Button styles
		$css .= '}';


		//---------------- Start Popup Button styles
		$css .= ".cu-popup-button { ";
		// Button colors
				$css .= "color: " . $opts['btn_text_color'] . "; "
					. "background-color: " . $opts['btn_bg_color'] . "; ";

		// end Popup Button styles
		$css .= '}';

		//---------------- Start Popup Hover Button styles
		$css .= ".cu-popup-button:hover,
				 .cu-popup-button:focus,
				 .cu-popup-button:active { ";

				// colors
				$css .= "color: " . $opts['btn_hover_text_color'] . "; "
					. "background-color: " . $opts['btn_hover_bg_color'] . "; ";

		// end Popup Hover Button styles
		$css .= '}';

		return $css;
	}
}

?>
